#! /usr/bin/env python
# -*- coding: utf-8 -*-
# Author: FCST Development Team
# License: TBD
# Copyright (c) 2012, TBD
"""
This module conatins a simple parser for the Dakota tabular results files.
Currrently only the cell voltage as input variable is supported.
Simple plotting of the data is provided as well.

Input:
    - Datafile, e.g. "dakota_tabular.dat"
      Format:
      %eval_id         V_cell       obj_fn_1       obj_fn_2       obj_fn_3 
       1             -1 -0.0002062952883 -2.934964907e-10 2.935131854e-10 
       2   -0.981372549 -0.0003800131988 -5.404900986e-10 5.405464135e-10 
       3   -0.962745098 -0.0006991604748 -9.940738811e-10 9.940738808e-10 
       4  -0.9441176471 -0.001283479196 -1.823389488e-09 1.823389487e-09 
       5  -0.9254901961 -0.002346731054 -3.32900551e-09 3.329005509e-09 
       6  -0.9068627451 -0.004260653874 -6.02802765e-09 6.028027651e-09 
       7  -0.8882352941 -0.007643260285 -1.076310996e-08 1.076310996e-08 
       8  -0.8696078431 -0.01344925073 -1.87865572e-08 1.878655721e-08 
       9  -0.8509803922 -0.02299931512 -3.170093075e-08 3.170093098e-08 
      10  -0.8323529412 -0.03787137823 -5.112308884e-08 5.112309389e-08 
      11  -0.8137254902 -0.05967012649 -7.811773346e-08 7.811773346e-08 
      12  -0.7950980392  -0.0898313972 -1.12697353e-07 1.12697353e-07 
      13  -0.7764705882  -0.1295823008 -1.536054117e-07 1.536054122e-07 
"""

# Basic imports:
import sys, os, shutil
import glob
import re
import commands

from StringIO import StringIO # Create stringlike file objects

# Import scientific packages:
import pylab as pl
import scipy as sp
import scipy.io
import scipy.interpolate
import scipy.optimize

fig_width_pt        = 246.0         # Get this from LaTeX using \showthe\columnwidth
inches_per_pt   = 1.0/72.27                 # Convert pt to inch
fig_width_cm        = 30.0
inches_per_cm   = 0.393
golden_mean = (pl.sqrt(5)-1.0)/2.0             # Aesthetic ratio
fig_width = fig_width_cm*inches_per_cm      # width in inches
fig_height = fig_width*golden_mean          # height in inches
fig_size =  [fig_width,fig_height]
params = {
            'backend': 'ps',
#           'backend': 'svg',
            'axes.labelsize': 18,
            'text.fontsize': 18,
            'titlesize': 24,
            'legend.fontsize': 16,
            'xtick.labelsize': 16,
            'ytick.labelsize': 16,
            'text.usetex': True,
            'figure.figsize': fig_size
        }
pl.rcParams.update(params)

class DakotaTabularParser(object):
    r"""
    Postprocessor for tabular data generated by dakota
    
    The default assumption at the moment is polarisation data.
    
    Parameters
    ----------
    :param filename: Filename of the tabular file to be parsed
    :param indent: Indentation
    :type filename: string
    :type indent: string
    :return: None
    :rtype: None
    
    Examples
    --------
    >>> import mkPostDakotaPlots as POST
    >>> test=POST.DakotaTabularParser("dakota_tabular.dat")
    
    .. warning:: This docuemtation is not yet finished.
    
    .. note::

    .. todo:: Finish documentation
        
    """
    def __init__(self, indent="\t",filename="dakota_tabular.dat",
                 responsenames={"obj1":"current"}):
        r"""
        Constructor of the class
        - Reads and parses datafile
        """
        self._indent=indent
        
        print indent,"="*40
        print indent,"= Dakota tabular file parser ="
        print indent,"-"*40
        
        print indent,"= - Parse body of the data file (1st row skipped)"
        print indent,"=   Filename: ", filename
        self.myData=sp.genfromtxt(fname=filename,names=True)
        print indent,"= - The following columns have been detected"
        print indent,"=   ", self.myData.dtype.names
        
        self.DictParameters={}
        self.DictObjective={};iter1=0
        
        for item in self.myData.dtype.names:
            if item.find("obj") != -1:
                print indent, "=\t - Response detected: ", item
                if sp.size(responsenames)>iter1:
                    print indent, "=\t   - Response identified as: ", responsenames[iter1]
                    self.DictObjective[responsenames[iter1]] = item
                else:
                    print indent, "=\t   - Response not identified"
                iter1=iter1+1
            if item.find("obj") == -1:
                print indent, "=\t - Parameter detected: ", item
                self.DictParameters[item]=item
                
        self.DictAll=self.DictParameters.copy()
        self.DictAll.update(self.DictObjective)
        
    def plotSinglePolPowerCurve(self,fig=None,basename="test",label="1"):
        
        print self._indent, "="*50
        print self._indent, "= Plot Polarisation curve "
        print self._indent, "=   Parameter: - 'U_cell'  "
        print self._indent, "=   Response:  - 'current'"
        print self._indent, "=","-"*49
        
        if (self.DictAll.has_key('U_cell')==False) and (self.DictAll.has_key('current')==False):
            print self._indent, " Error: valid entry missing"
            return
        if fig==None:
            fig=pl.figure()
        
        host=fig.add_subplot(111)
        par1=host.twinx()

        p1, = host.plot(-self.myData[self.DictAll["current"]],
                        -self.myData[self.DictAll["V_cell"]],'bo-',label='Polcurve')
        p2, = par1.plot(-self.myData[self.DictAll["current"]],
                        self.myData[self.DictAll["V_cell"]]*self.myData[self.DictAll["current"]],
                        'go--',label='Power Density')

        host.set_xlabel("Current Density [$A/cm^2$]")
        host.set_ylabel("Cell Voltage [$V$]")
        host.set_title("Polarization Curve" + " ( " + basename + " = " + label + " )")
        par1.set_ylabel("Power Density [$W/cm^2$]")
        
        host.yaxis.label.set_color(p1.get_color())
        host.set_ylim(0, 1)
        host.tick_params(axis='y',colors=p1.get_color())
        par1.yaxis.label.set_color(p2.get_color())
        par1.tick_params(axis='y',colors=p2.get_color())
        
        lines = [p1, p2]
        host.legend(lines, [l.get_label() for l in lines],loc="upper left")
        
        host.axvline(x=0.1,linewidth=1, color='k')
        host.axvline(x=1.9,linewidth=1, color='k')
        host.axvline(x=2.4,linewidth=1, color='k')
        host.axhline(y=0.622,linewidth=1, color='k')
        host.axhline(y=0.670,linewidth=1, color='k')    
        host.axhline(y=0.840,linewidth=1, color='k') 
        host.plot([0.1,1.9,2.4],[0.840,0.670,0.622],'ko')
        
        
        pl.savefig("FigPolcurve_" + basename + "_"+label+".png")
        
    def plotSinglePolLinLogCurve(self,fig=None,basename="test",label="1"):
        print self._indent, "="*50
        print self._indent, "= Plot Polarisation curve "
        print self._indent, "=   Parameter: - 'U_cell'  "
        print self._indent, "=   Response:  - 'current'"
        print self._indent, "=","-"*49
        
        if fig==None:
            fig=pl.figure()
        
        host=fig.add_subplot(111)
        p1, = host.semilogx(-self.myData[self.DictAll["current"]],
                            -self.myData[self.DictAll["V_cell"]],'bo-',label='Polcurve')
        host.set_xlabel("Current Density [$A/cm^2$]")
        host.set_ylabel("Cell Voltage [$V$]")
        pl.title("Polarization Curve" + " ( " + basename + " = " + label + " )")
        pl.savefig("FigPolcurveSemiLogX_" + basename + "_"+label+".png")

    def plotSingleWaterCrossoverCurve(self,fig=None,basename="test",label="1"):
        
        print self._indent, "="*50
        print self._indent, "= Plot Membrane Water Crossover "
        print self._indent, "=   Response:  - 'current'  "
        print self._indent, "=   Response:  - 'membrane_water_c'"
        print self._indent, "=   Response:  - 'membrane_water_a' [optional]"
        print self._indent, "=","-"*49
        
        if (self.DictAll.has_key('membrane_water_c')==False) and (self.DictAll.has_key('current')==False):
            print self._indent, " Error: valid entry missing"
            return

        if fig==None:
            fig=pl.figure()
        host=fig.add_subplot(111)

        p1, = host.plot(-self.myData[self.DictAll["current"]],
                        -self.myData[self.DictAll["membrane_water_c"]],
                        'go-',label='Cathode')
        if (self.DictAll.has_key('membrane_water_a')==True):
            p1, = host.plot(-self.myData[self.DictAll["current"]],
                        self.myData[self.DictAll["membrane_water_a"]],
                        'bx--',label='Anode')
                        
        pl.fill_between(-self.myData[self.DictAll["current"]],
                     -self.myData[self.DictAll["membrane_water_c"]],
                     -self.myData[self.DictAll["current"]]*0,
                     where=-self.myData[self.DictAll["membrane_water_c"]]<=0,
                     facecolor='blue' , interpolate=True)
        pl.fill_between(-self.myData[self.DictAll["current"]],
                     -self.myData[self.DictAll["membrane_water_c"]],
                     -self.myData[self.DictAll["current"]]*0,
                     where=-self.myData[self.DictAll["membrane_water_c"]]>=0,
                     facecolor='green' , interpolate=True)

        host.set_xlabel("Current Density [$A/cm^2$]")
        host.set_ylabel("Water Crossover [$xxx$]")
        host.set_title("Membrane Water Crossover" + " ( " + basename + " = " + label + " )")
        
        host.yaxis.label.set_color(p1.get_color())
        #host.set_ylim(0, 1)
        host.tick_params(axis='y',colors=p1.get_color())
        
        pl.savefig("FigWaterCrossover_" + basename + "_"+label+".png")

def main():
    print "-"*50
    print "= - Parse Commandline "
    # Import command line option parser:
    import argparse
    parser = argparse.ArgumentParser(
        description=
            'Program to process dakota tabular output files. ' \
            'Multiple files can be parsed at the same time. ' \
            'Implemented plots are:\n' \
            ' - Polarisation curve (with power density) \n' \
            ' - Membrane water crossover curve'
            ,formatter_class=argparse.RawDescriptionHelpFormatter)
    parser.add_argument('filenames', type=str, nargs='+',
                   help='filenames to be processed')
    parser.add_argument('--basename', type=str, nargs='+',default='testrun',
                   help='a basename to be used for display or storage')              
    parser.add_argument('--label', type=str, nargs='+',default='',
                   help='space separated list of labels (size has to match the number of processed files)')
    parser.add_argument('--responses', type=str, nargs='+',
                   default=['current','membrane_water_c','membrane_water_a'],
                   help='space separated list of dakota responses')
    args = parser.parse_args()
    print args.filenames
    print "=   - Files to be processed:"
    for filename in args.filenames:
        print "=     ", filename  
        
    print "=   - Basename:  ", args.basename[0]
    print "=   - Labels  :  ", args.label
    print "=   - Responses: ", args.responses
    
    Datasets={}
    ii=0
    for filename in args.filenames:
        print "=   - Process file ", filename
        f1=pl.figure()
        f2=pl.figure()
        Datasets[filename] = DakotaTabularParser(filename=filename,
                        responsenames=args.responses)
        Datasets[filename].plotSinglePolPowerCurve(fig=f1,
                        basename=args.basename[0],
                        label=args.label[ii])
        Datasets[filename].plotSingleWaterCrossoverCurve(fig=f2,
                        basename=args.basename[0],
                        label=args.label[ii])
        #Datasets[filename].plotSinglePolLinLogCurve(fig=f2,basename=args.basename[0],label=args.label[ii])
        ii=ii+1
    pl.show()
        
    # Emergency plot lines. This has to be done properly:
    fig=pl.figure()
    host=fig.add_subplot(111)
    pl.hold(True)
    ii=0
    for filename in args.filenames:
        print filename
        myData=Datasets[filename].myData.copy()
        pl.plot(-myData['obj_fn_1'],-myData["V_cell"],label=args.label[ii],linewidth=2)
        pl.title("Polarization Curve" + " ( " + args.basename[0] + " )")
        ii=ii+1
    
    
    pl.legend()
    host.axvline(x=0.1,linewidth=1, color='k')
    host.axvline(x=1.9,linewidth=1, color='k')
    host.axvline(x=2.4,linewidth=1, color='k')
    host.axhline(y=0.622,linewidth=1, color='k')
    host.axhline(y=0.670,linewidth=1, color='k')    
    host.axhline(y=0.840,linewidth=1, color='k') 
    pl.plot([0.1,1.9,2.4],[0.840,0.670,0.622],'ko')
    
    host.set_xlabel("Current Density [$A/cm^2$]")
    host.set_ylabel("Cell Voltage [$V$]")
    pl.savefig("FigPolcurve_" + args.basename[0] + "_Summary.png")
    
    fig=pl.figure()
    host=fig.add_subplot(111)
    pl.hold(True)
    ii=0
    for filename in args.filenames:
        print filename
        myData=Datasets[filename].myData.copy()
        pl.semilogx(-myData['obj_fn_1'],-myData["V_cell"],label=args.label[ii])
        ii=ii+1
    pl.legend()
    host.set_xlabel("Current Density [$A/cm^2$]")
    host.set_ylabel("Cell Voltage [$V$]")
    pl.savefig("FigPolcurveSemiLogX_" + args.basename[0] + "_Summary.png")
    pl.show()
    return Datasets
    
  
if __name__    =="__main__":
    main()    