#! /usr/bin/env python
# -*- coding: utf-8 -*-
# Author: FCST Development Team
# License: TBD
# Copyright (c) 2012, TBD
r"""
*************************************************************************
:mod:`PythonFCST.util.parsers`: Parsers for different file formats
*************************************************************************

.. module:: PythonFCST.util.parsers

Contents
--------
The PythonFCST package imports all the functions from the top level modules. 
 
Import
------
>>> import PythonFCST as fcst
>>> tmp=fcst.BASE.testinheritance()


Submodules
----------
::

 None                            --- No subpackages at the moment


"""



from baseclass import fcstobject
import scipy as sp
import pylab as pl
import scipy.io
import scipy.interpolate
import scipy.optimize
from itertools import cycle

import sys, os, shutil
import glob
import re
from StringIO import StringIO # Create stringlike file objects

_pvdheader=r"""<?xml version="1.0"?>
<VTKFile type="Collection" version="0.1">
  <Collection>
"""

_pvdfooter=r"""  
   </Collection>
</VTKFile>
"""


class read_csv(fcstobject):
    r"""
    .. class::`PyFCell.util.parsers.read_csv` -- Importer for CSV type data
    
    
    Abstraction for scipy.genfromtext with some nice output.
    
    Parameters
    ----------    
    
    fname : string
        Name of the file to be imported
    delimiter : string (default="'")
        Delimiter argument for the textfile reader.

    Attributes
    ----------
    
    
    """
    def __init__(self,fname="test.csv",delimiter=",",comments="#",
                 names=True,dtype=None,**kwargs):
        super(read_csv,self).__init__(**kwargs)
        self._logger.debug("Initialize cvsreader object")
        self._data=sp.genfromtxt(fname,delimiter=delimiter,names=names,
                                 dtype=dtype,deletechars='')
    def get_data(self):
        return self._data
        
    def print_all_column_info(self):
        print "="*50
        print "= Size", self._data.shape
        for name in self._data.dtype.names:
            print name, "\t\t", self._data[name].dtype
        print "="*50
    def print_column_types(self,column=None):
        temp=sp.unique(self._data[column])
        print "The unique entries of column ", column, "are:"  
        for tt in temp:
            print "\t ", tt
    def get_filtered_data(self,column=None,criterion=None):
        indices = sp.where(self._data[column]==criterion)
        return self._data[indices]


class read_polcurve(fcstobject):
  r"""
  Importer and plotter for polcurves generated by fcst without dakota.
  
  Postprocessor for tabular polcarization curve data produced by fcst.
  This code assumes the header strings to be predetermined.
  
  Parameters
  ----------
  fname : string (default "dakota_tabular.dat")
        Name of the file to be imported. This file has to be  a fcst
        tabular output file.
        
        Format::
        
	  # ====================================================
	  # OpenFCST: Fuel cell simulation toolbox 
	  # ====================================================
	  # Polarization curve data :polarization_curve.dat
	  Cell voltage [V]        Cathode current [A/cm2]
	  0.9     0.007674716442 
	  0.8     0.09141826831 
	  0.7     0.3239937606
	  0.6     0.7566928344 
	  0.5     1.458893632
	  0.4     2.18243

        
  
  """
  def __init__(self,fname='polarization_curve.dat', **kwargs):
      super(read_polcurve,self).__init__(**kwargs)
      try:
          import pandas as pd
      except ImportError:
          self._logger.error('Import Error: pandas could not be loaded')
          raise ImportError('Cannot import pandas, please install the module')
      
      self._df = pd.read_csv(fname, comment='#',delimiter='\t')
      print 'Column header: ', self._df.columns.tolist()
      print 'Rows:          ', self._df.shape[0]
          
    
  def plot_polcurve(self,fig=None, label="Dataset 1", 
                      format_pol = "b-", format_power = "g--",
                      flag_power=True, flag_guides=True):
        r"""
        Plot polarisation curve
        """
        
        # Setup figure environment
        if fig==None:
            fig=pl.figure()
            
            host=fig.add_subplot(111)
            host.set_xlabel("Current Density [$A/cm^2$]")
            host.set_ylabel("Cell Voltage [$V$]")
            # host.set_title("Polarization Curves")
            host.yaxis.label.set_color('b')
            host.set_ylim(0, 1)
            host.tick_params(axis='y',colors='b')
            
            if flag_power:
                par1=host.twinx()
                par1.set_ylabel("Power Density [$W/cm^2$]")
                par1.yaxis.label.set_color('g')
                par1.tick_params(axis='y',colors='g')                    
                
            lines=[]
        
        else:
            host = fig.get_axes()[0]
            if flag_power:
                par1 = fig.get_axes()[1]
            lines=[]
        
        # Check for necessary column headers        
        
        p1, = host.plot(self._df['Cathode current [A/cm2]'],
                        self._df['Cell voltage [V]'],
                        format_pol,
                        # label='Polcurve ( ' + label + ' ) ',
                        label=label,
                        linewidth=2)
        # Add anode plot
        
    
        
        #lines.append[p1]
                        
        if flag_power: 
            p2, = par1.plot(self._df['Cathode current [A/cm2]'],
                             self._df['Cathode current [A/cm2]']*self._df['Cell voltage [V]'],
                            format_power,
                            label='Power Density ( ' + label + " )")
            
            #lines.append(p2)
        
        #host.legend(loc="upper left",ncol=2)
        legend = host.legend(loc='upper center', bbox_to_anchor=(0.5, 1.10),
                             ncol=4,fancybox=True, shadow=True)
        legend.draggable()
        #host.legend(lines, [l.get_label() for l in lines],loc="upper left")
        
        if flag_guides:
            host.axvline(x=0.1,linewidth=1, color='k')
            host.axvline(x=1.9,linewidth=1, color='k')
            host.axvline(x=2.4,linewidth=1, color='k')
            host.axhline(y=0.622,linewidth=1, color='k')
            host.axhline(y=0.670,linewidth=1, color='k')    
            host.axhline(y=0.840,linewidth=1, color='k') 
            host.plot([0.1,1.9,2.4],[0.840,0.670,0.622],'ko')
        
        return fig        
  
  
default_responses={
    'obj_fn_1':'Cell Current (Cathode)',
    'obj_fn_2':'Temperature (Max)',
    'obj_fn_3':'Cell Current (Anode)'
    }    
default_parameters={
    'eval_id': 'Evaluation ID',
    'DV_0' : 'Cell Voltage'
    }
    


class read_dakota(fcstobject):
    
    r"""
    .. class::`PyFCell.util.parsers.read_dakota` -- Importer for dakota tabular dat.
    
    
    Postprocessor for tabular data generated by dakota
    
    The default assumption at the moment is polarisation data.
    
    Parameters
    ----------
    
    fname : string (default "dakota_tabular.dat")
        Name of the file to be imported. This file has to be  a dakota
        tabular output file.
        
        Format::
        
            %eval_id         DV_0       obj_fn_1       obj_fn_2       obj_fn_3 
            1             -1 -0.0002062952883 -2.934964907e-10 2.935131854e-10 
            2   -0.981372549 -0.0003800131988 -5.404900986e-10 5.405464135e-10 
            3   -0.962745098 -0.0006991604748 -9.940738811e-10 9.940738808e-10 
            4  -0.9441176471 -0.001283479196 -1.823389488e-09 1.823389487e-09 
            5  -0.9254901961 -0.002346731054 -3.32900551e-09 3.329005509e-09 
            6  -0.9068627451 -0.004260653874 -6.02802765e-09 6.028027651e-09 
            7  -0.8882352941 -0.007643260285 -1.076310996e-08 1.076310996e-08 
            8  -0.8696078431 -0.01344925073 -1.87865572e-08 1.878655721e-08 
            9  -0.8509803922 -0.02299931512 -3.170093075e-08 3.170093098e-08 
        
    parameters : dictionary (default=" ")
        Translation for the table headings for parameters.
    responses  : dictionary ()
        Translation for the table headings for responses.
    
    
    Examples
    --------
    >>> import mkPostDakotaPlots as POST
    >>> test=POST.DakotaTabularParser("dakota_tabular.dat")
    
    .. warning:: This docuemtation is not yet finished.
    
    .. note::

    .. todo:: Finish documentation
        
    
    
    """
    def __init__(self,fname="dakota_tabular.dat",parameters=default_parameters,responses=default_responses,**kwargs):
        super(read_dakota,self).__init__(**kwargs)
        
        indent = "|DAKOTA|"
        self._logger.debug("Initialize Dakota reader object")        
        self._logger.debug("\tParse dakota tabular file")
        data=sp.genfromtxt(fname=fname,names=True)
        
        names = data.dtype.names
        names_new = []
        
        self.DictParameters=parameters
        self.DictResponses=responses;
        self._design_variables=[]
        self._objectives=[]
        self._datadict={}
        self._keys={}
        
        outputstr="Identify column headers\n"
        
        for item in data.dtype.names:
            if self.DictResponses.has_key(item):
                outputstr = outputstr + "\t   - Response " + item + "identified as: " +self.DictResponses[item] +'\n'
                self._datadict[self.DictResponses[item]] = data[item]
                names_new.append(self.DictResponses[item])
                self._objectives.append(self.DictResponses[item])
            elif self.DictParameters.has_key(item):
                outputstr = outputstr + "\t   - Parameter " + item + "identified as: " + self.DictParameters[item] + '\n'
                self._datadict[self.DictParameters[item]] = data[item]
                names_new.append(self.DictParameters[item])
                self._design_variables.append(self.DictParameters[item])
            else:
                self._logger.warning("Dakota tabular header " + item + " not identified")
                names_new.append(item)
                
        data.dtype.names=names_new
        self._data=data
        
        self.set_names_default()        
        self.delete_invalid_data()
        
        self._logger.info(outputstr)
        
    def delete_invalid_data(self,value=0.0):
        r"""
        .. todo:: The explicite dependency on cathode current needs to be removed
        """
        rows=sp.where(self._data[self._objectives[0]]==value)
        self._logger.warning('Deleting invalid data rows: '+str(rows))
        sp._data=sp.delete(self._data,rows,axis=0)
        for key in self._datadict.keys():
            self._datadict[key] = sp.delete(self._datadict[key],rows,axis=0)
        
        
    def get_data(self):
        return self._data
        
    def set_names_default(self):
        self.set_names_polcurve()
        self.set_names_crossover()
        
    def set_names_polcurve(self,
                           cell_voltage='Fuel cell data>>Operating conditions>>Voltage cell',
                           current_cathode='cathode_current',
                           current_anode='anode_current'):
        
        if self._datadict.has_key(cell_voltage):
            self._keys["cell_voltage"]=cell_voltage
            self._logger.info("Parameter " + cell_voltage + " identified as cell_voltage")
        if self._datadict.has_key(current_cathode):
            self._keys["current_cathode"]=current_cathode
            self._logger.info("Parameter " + current_cathode + " identified as current_cathode")
        if self._datadict.has_key(current_anode):
            self._keys["current_anode"]=current_anode
            self._logger.info("Parameter " + current_anode + " identified as current_anode")
            
    def set_names_crossover(self,
                            water_cathode='water_cathode',
                            water_anode='water_anode'):
        if self._datadict.has_key(water_cathode):
            self._keys["water_cathode"]=water_cathode
        if self._datadict.has_key(water_anode):
            self._keys["water_anode"]=water_anode
            
    def set_names_thermal(self,
                          max_temp      = 'max_temperature',
                          react_cathode = 'cathode_reaction_heat',
                          react_anode   = 'anode_reaction_heat'
                          ):
        if self._datadict.has_key(max_temp):
            self._keys["max_temp"]=max_temp
        if self._datadict.has_key(react_cathode):
            self._keys["heat_react_c"]=react_cathode
        if self._datadict.has_key(max_temp):
            self._keys["heat_ract_a"]=react_anode
        
            
    def set_names_heatloss(self):
        print "Hallo"
            
    
        
    def plot_polcurve(self,fig=None, label="FCST test", 
                      format_pol = "b-", format_power = "g--",
                      flag_power=True, flag_guides=True):
        if (self._keys.has_key('cell_voltage')==False) and (self._keys.has_key('current_cathode')==False):
            print self._indent, " Error: valid entry missing"
            return pl.figure()
        if fig==None:
            fig=pl.figure()
            
            host=fig.add_subplot(111)
            host.set_xlabel("Current Density [$A/cm^2$]")
            host.set_ylabel("Cell Voltage [$V$]")
            # host.set_title("Polarization Curves")
            host.yaxis.label.set_color('b')
            host.set_ylim(0, 1)
            host.tick_params(axis='y',colors='b')
            
            if flag_power:
                par1=host.twinx()
                par1.set_ylabel("Power Density [$W/cm^2$]")
                par1.yaxis.label.set_color('g')
                par1.tick_params(axis='y',colors='g')                    
                
            lines=[]
        
        else:
            host = fig.get_axes()[0]
            if flag_power:
                par1 = fig.get_axes()[1]
            lines=[]
            
            
        
        
        col_voltage=self._keys['cell_voltage']
        col_current_c = self._keys['current_cathode']        
        
        
        p1, = host.plot(-self._datadict[col_current_c],
                        -self._datadict[col_voltage],format_pol,
                        # label='Polcurve ( ' + label + ' ) ',
                        label=label,
                        linewidth=2)
                        
        if self._keys.has_key('current_anode'):
            col_current_a = self._keys['current_anode']
            host.plot(-self._datadict[col_current_c],
                      -self._datadict[col_voltage],'k.')
    
        
        #lines.append[p1]
                        
        if flag_power:
            
            p2, = par1.plot(-self._datadict[col_current_c],
                             self._datadict[col_current_c]*self._datadict[col_voltage],
                            format_power,
                            label='Power Density ( ' + label + " )")
            
            #lines.append(p2)
        
        #host.legend(loc="upper left",ncol=2)
        legend = host.legend(loc='upper center', bbox_to_anchor=(0.5, 1.10),
                             ncol=4,fancybox=True, shadow=True)
        legend.draggable()
        #host.legend(lines, [l.get_label() for l in lines],loc="upper left")
        
        if flag_guides:
            host.axvline(x=0.1,linewidth=1, color='k')
            host.axvline(x=1.9,linewidth=1, color='k')
            host.axvline(x=2.4,linewidth=1, color='k')
            host.axhline(y=0.622,linewidth=1, color='k')
            host.axhline(y=0.670,linewidth=1, color='k')    
            host.axhline(y=0.840,linewidth=1, color='k') 
            host.plot([0.1,1.9,2.4],[0.840,0.670,0.622],'ko')
        
        return fig        
    
    def plot_max_temp(self, fig=None, label='Test', format_plot='r-'):
        if (self._keys.has_key('max_temp')==False):
            print self._logger.error("Valid entry missing for key 'max_temp'")
            return
        if fig==None:
            fig=pl.figure()
            host=fig.add_subplot(111)
            host.set_xlabel("Current Density [$A/cm^2$]")
            host.set_ylabel("$\Delta T$ [$K$]")
        else:
            host=fig.get_axes()[0]
            
        current = self._datadict[self._keys['current_cathode']]
        maxtemp = self._datadict[self._keys['max_temp']]
                
        host.plot(-current, maxtemp-maxtemp[0], format_plot,label=label)
        
        legend = host.legend(loc='upper center', bbox_to_anchor=(0.5, 1.10),
                             ncol=4,fancybox=True, shadow=True)
        legend.draggable()
        return fig
        
    def plot_thermal_losses(self):
        if (self._keys.has_key('heat_react_c')==False) and (self._keys.has_key('heat_react_a')==False):
            print self._logger.error("Valid entry missing for key 'max_temp'")
            return
        if fig==None:
            fig=pl.figure()
            host=fig.add_subplot(111)
            host.set_xlabel("Current Density [$A/cm^2$]")
            host.set_ylabel("$Losses$ [$???$]")
        else:
            host=fig.get_axes()[0]
            
        current = self._datadict[self._keys['current_cathode']]
        react_c = self._datadict[self._keys['heat_react_c']]
                
        host.plot(-current, react_c, format_plot,label=label)
        
        legend = host.legend(loc='upper center', bbox_to_anchor=(0.5, 1.10),
                             ncol=4,fancybox=True, shadow=True)
        legend.draggable()
        return fig
        
    def plot_water_crossover(self,fig=None,label='Test', 
                             format_cathode ='r-', format_anode='r--'):
        if (self._keys.has_key('water_anode')==False) and (self._keys.has_key('water_cathode')==False):
            self._logger.warning("Keys to plot water crossover missing - skip")
            return
        if fig==None:
            fig=pl.figure()
            host=fig.add_subplot(111)
            host.set_xlabel("Current Density [$A/cm^2$]")
            host.set_ylabel("Water Crossover (Anode to Cathode) [$????$]")
            #host.set_title("Membrane Water Crossover")
        else:
            host=fig.get_axes()[0]
        
        current=self._datadict[self._keys['current_cathode']]
        water_anode =   self._datadict[self._keys['water_anode']]
        water_cathode = self._datadict[self._keys['water_cathode']]
        
        host.plot(-current,-water_cathode,format_cathode,label=label)
        host.plot(-current, water_anode,format_anode)
        
        #host.legend(loc="upper left")
        legend = host.legend(loc='upper center', bbox_to_anchor=(0.5, 1.10),
                             ncol=4,fancybox=True, shadow=True)
        legend.draggable()
        return fig
        
        
class ReadOptPrm(fcstobject):
    r"""
    .. class::`PyFCell.util.parsers.read_opt_prm` -- Parse the optimization parameter file
    
    
    Abstraction for scipy.genfromtext with some nice output.
    
    Parameters
    ----------    
    
    fname : string
        Name of the file to be parsed


    Attributes
    ----------
    
    
    """
    def __init__(self,fname="test.csv",dtype=None,**kwargs):
        super(ReadOptPrm,self).__init__(**kwargs)
        self._logger.info("Initialize cvsreader object")
        
        self._logger.info('Open file '+fname)
        self._datafile=open(fname)
    
    def find_section(self,sectname="subsection Design Variables"):
        self._logger.info('Find ' + sectname)
        self._datafile.seek(0)
        line = self._datafile.readline()
        while line:
            line=line.lstrip().rstrip()
            if sectname in line:
                self._logger.info("\tsubsection found")
                return line
            line = self._datafile.readline()
            
    
    def find_design_variables(self):
        
        line=self.find_section(sectname="subsection Design Variables")
        
        
        pattern = r"set DV_(?P<num>\d*)_name = (?P<name>.*)"
        pattern_c = re.compile(pattern)
        
        parameters={'eval_id': 'Evaluation ID'}
        
        while line:
            line=line.lstrip().rstrip()
            if "set num_design_variables" in line:
                print line
            match = pattern_c.match(line)
            if match:
                self._logger.info("\tFound Design Variable "+ match.group('num')
                                    +' with value '+match.group('name') )
                key = "DV_"+match.group('num')
                name = match.group('name')
                #if "Fuel cell data:Operating conditions:Voltage cell" in name:
                #    name="cell_voltage"
                parameters[key] = name
            line = self._datafile.readline()
        
        return parameters
    
    def find_objective_functionals(self):
        line=self.find_section(sectname="subsection Responses")
        
        pattern = r"set RESP_(?P<num>\d*)_name.*= (?P<name>.*)"
        pattern_c = re.compile(pattern)
        
        numpattern = r"set num_objectives = (?P<num>[0-9]*).*"
        numpattern_c = re.compile(numpattern)
        numobj = 0        
        parameters={}
        
        
        while line:
            line=line.lstrip().rstrip()
            
            if "set num_objectives" in line:
                match = numpattern_c.match(line)
                numobj = int(match.group('num'))
                
            match = pattern_c.match(line)
            if match:
                self._logger.info("\tFound Objective "+ match.group('num')
                                    +' with value '+match.group('name') )
                if numobj == 1:
                    key = "obj_fn"
                else:
                    key = "obj_fn_"+str( int(match.group('num'))+1 )
                    
                name=match.group('name')
                #if name=="current":
                #    name="cathode_current"
                parameters[key] = name
            line = self._datafile.readline()
        
        return parameters         
        
        
class CreatePvd(fcstobject):
    r"""
    .. class::`PyFCell.util.parsers.CreatePVD` -- Creates a paraview pvd file
    
    
    Creates a paraview .pvd file from any structured set of xml vtk files.
    
    ..note: Paraview cannot open legacy .vtk files with this method
    
    Parameters
    ----------    
    
    pattern : string (default: "fuel\\_cell\\_solution\\_DataFile\\_(?P<order>[0-9]+)\\_Cycle\\_2\\.vtu")
        Name of the file to be imported
    fname : string (default="paraview.pvd")
        Delimiter argument for the textfile reader.

    Examples
    --------
    
    Please add an example here
        
    
        
    """
    _filelist = []    
    def __init__(self,pattern="fuel\\_cell\\_solution\\_DataFile\\_(?P<order>[0-9]+)\\_Cycle\\_2\\.vtu",fname='paraview.pvd',group='order',**kwargs):
        super(CreatePvd,self).__init__(**kwargs)
        self._logger.warning('Pattern: '+pattern)
        self._regex=re.compile(pattern)
        self._group=group
        
        self._out = open(fname,'w')
        
    def __del__(self):
        self._out.close()
        
    def find_files(self):
        for filename in os.listdir('.'):
            match=self._regex.match(filename)
            if match:
                self._filelist.append(filename)
        self._logger.info('Found '+str(len(self._filelist))+'files matching pattern')
    
    def sort(self):
        def grp(txt):
            r=self._regex.match(txt)
            return r.group(self._group) if r else '&'
        
        self._filelist.sort(key=grp)
        
    def print_files(self):
        for filename in self._filelist:
            print filename
            
    def write_pvd(self):
        self._out.write(_pvdheader)
        for filename in self._filelist:
            self._out.write(r'''      <DataSet timestep="''')
            match=self._regex.match(filename)
            match=str(int((match.group(self._group))))
            self._out.write(match)
            self._out.write('"')
            # out.write(r''' group="test" part="0"''')
            self._out.write(r''' file="''')
            self._out.write(filename)
            self._out.write(r'''"/>''')
            self._out.write('\n')
        self._out.write(_pvdfooter)
        self._logger.info('Wrote file '+self._out.name)
        
    def run(self):
        self.find_files()
        self.sort()  
        self.print_files()
        self.write_pvd()
    
    
def main():
    print "Hallo"
    import PythonFCST as fcst
    parser=fcst.util.parsers.read_dakota(fname='dakota_tabular.dat')
    return parser
if __name__    =="__main__":
    parser=main()  